"""------------------------------------------------------------------------------------------
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%  Code for replication of results in the paper:
%%%%  Denotative and Connotative Control of Uncertainty: A Computational Dual Process Model
%%%%  Authors: Jesse Hoey (University of Waterloo),  
%%%%           Neil MacKinnon (University of Guelph),
%%%%           Tobias Schroeder  (Potdsam University of Applied Sciences)
%%%%  Copyright: the authors, 2021
%%%%  For research purposes only.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% This file:  Python 2.x code to replicate fairness and PCS (juror) results from sections 5.1 and 5.3
%%%%% USAGE: python actsimulator.py  -h  
%%%%%% displays a help message with options
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
----------------------------------------------------------------------------------------------"""
import os
import math
import sys
import re
import copy
import getopt
import numpy as NP
import itertools
import time
import random

#-------------------------------------------------------------------------------
#  dictionaries of identities, behaviours and modifiers (emotions)
#  and dynamics equations 
#-------------------------------------------------------------------------------
fifname = "fidentities.dat"
fbfname = "fbehaviours.dat"
fmfname = "fmodifiers.dat"

## impression formation (dynamics) equations 
temfname = "tdynamics-male.dat"
teffname = "tdynamics-female.dat"

## emotion dynamics equations
emfname=os.path.join(os.path.dirname(__file__), "temotions-male.dat")
effname=os.path.join(os.path.dirname(__file__), "temotions-female.dat")

#------------------------------------------------------------------------------
# book-keeping, random stuff
#------------------------------------------------------------------------------

#normally this is ["e","p","a"]
epa=["e","p","a"]
#this is usually  ["a","b","c"]
sentvals=["a","b","c"]


#for discretising the sentiment range - not really used anymore
sentiment_levels=2
smin=-4.3
smax=4.3
sincr=(smax-smin)/sentiment_levels
sentiment_indices=xrange(sentiment_levels+1)
sentiment_bins=[x*sincr+smin for x in list(xrange(sentiment_levels+1))]

#create names for fundamentals and transients
fvars=[]
tvars=[]
for s in sentvals:
    for e in epa:
        tvars.append("T"+s+e)
        fvars.append("F"+s+e)


#-------------------------------------------------------------------------------
# open the input file for emotions
#-------------------------------------------------------------------------------
#which contains a binary number giving the terms in (fae,fap,faa,ea,ep,ea) where (ea,ep,ea) is the emotions vector
#followed by 3 coefficients
#the first row gives the constant d
#the next three rows give the 3x3 matrix E
#the next three rows give the 3x3 matrix R
#the final row gives the diagonal of the diagonal matrix Q
#for males
fb=file(emfname,"rU")
emot_d=map(lambda(z): float(z),fb.readline().split()[1:])
#print emot_d
emot_E=[]
for i in range(3):
    emot_E.append(map(lambda(z): float(z),fb.readline().split()[1:]))
#actually the transpose
emot_E = NP.array(emot_E).transpose()
#print emot_E
emot_R=[]
for i in range(3):
    emot_R.append(map(lambda(z): float(z),fb.readline().split()[1:]))
emot_R = NP.array(emot_R).transpose()
#the rest is the Q matrix - 
#this is read generally as follows: 
#Z100XXX means Q_e and multiples fa_e
#Z100010 means the coefficient of e_e multiples fa_p
#Z010XXX means Q_p and multiplies fa_p
#...etc
#we will assume only one fa and one e multiplier per matrix row
emot_Q = {}
for l in ['e','p','a']:
    emot_Q[l] = []
    for i in range(3):
        emot_Q[l].append([0.0,0.0,0.0])
x = fb.readline()
while x!="":
    y=x.split()
    #the f value to use is a 1 in this vector
    fvec=list(y[0])[4:7]
    fvecc=[w[0] for w in filter(lambda (z): z[1]=="1", zip(fvars[0:3],fvec))]
    #fvecc has a string like Fae in it - the third letter gives us which Q matrix this corresponds to
    #the tau dimension we assign to  is given by the position of the 1 in this vector
    evec = list(y[0])[1:4]
    emot_Q[fvecc[0][2]][evec.index('1')] = map(lambda(z): float(z),y[1:])
    x = fb.readline()
fb.close()

for l in ['e','p','a']:
    emot_Q[l] = NP.array(emot_Q[l]).transpose()

#-----------------------------------------------------------------------------
# helper functions for computing emotions and modified identities
#-----------------------------------------------------------------------------
def computeEmotion(fa,ta):
    normfact = copy.deepcopy(emot_E)
    #possibly don't do the ones that are all zeros here
    for (i,e) in zip([0,1,2],['e','p','a']):
        normfact += NP.diag([fa[i]]*3)*emot_Q[e]
    return NP.dot(NP.linalg.inv(normfact),(ta-NP.dot(emot_R,fa)-emot_d))
        
def computeModifiedIdentity(fa,felt_emotion):
    normfact = copy.deepcopy(emot_E)
    #possibly don't do the ones that are all zeros here
    for (i,e) in zip([0,1,2],['e','p','a']):
        normfact += NP.diag([fa[i]]*3)*emot_Q[e]
    return NP.dot(normfact,felt_emotion)+NP.dot(emot_R,fa)+emot_d
        

#---------------------------------------------------------------------------------------------------------------------------
#Builds some global constants describing dynamics (this is the matrix "M")
#---------------------------------------------------------------------------------------------------------------------------
#open the input file for dynamics of transients
#which contains a binary number giving the terms in tvars to be used
#followed by the 9-D multiplication matrix M row
# the tvars are ordered as:
# Ae, Ap, Aa, Be, Bp, Ba, Oe, Op, Oa,
#the result is e.g. tdyn-male which is an array of arrays, each of which
#contains a list of tvars terms followed by the matrix row
#for males
tdyn_male=[]
fb=file(temfname,"rU")
x=fb.readline()
while x!="":
    y=x.split()
    tvec=list(y[0])[1:]
    tvecc=[w[0] for w in filter(lambda (z): z[1]=="1", zip(tvars,tvec))]
    M=map(lambda(z): float(z),y[1:])
    tdyn_male.append([tvecc,M])
    x=fb.readline()
fb.close()
#for females
tdyn_female=[]
fb=file(teffname,"rU")
x=fb.readline()
while x!="":
    y=x.split()
    tvec=list(y[0])[1:]
    tvecc=[w[0] for w in filter(lambda (z): z[1]=="1", zip(tvars,tvec))]
    M=map(lambda(z): float(z),y[1:])
    tdyn_female.append([tvecc,M])
    x=fb.readline()
fb.close()


#-----------------------------------------------------------------------------
# other helper functions 
#-----------------------------------------------------------------------------
#true if mrow[0] has a variable that starts with v in it
def has_variable(mrow,v):
    return reduce(lambda xx,yy: xx or yy.startswith(v),mrow,False)

## gets a factor from impressions
def getFactor(indices, tau):
    return reduce(lambda x, i: x*tau[i], indices, 1)

#raw squared distance between two epa vectors
def raw_dist(epa1,epa2):
    dd = NP.array(epa1)-NP.array(epa2)
    return NP.dot(dd,dd)

#maps a sentiment label cact to an epa value using the dictionary fbeh
def mapSentimentLabeltoEPA(fbeh,cact):
    return map(lambda x: float (x), [fbeh[cact]["e"],fbeh[cact]["p"],fbeh[cact]["a"]])

#raw distance from an epa vector oval to a behaviour vector
def fdist(oval,fbehv):
    cd=[float(fbehv["e"]),float(fbehv["p"]),float(fbehv["a"])]
    return raw_dist(oval,cd)

#identify if at least one match exists between two lists of booleans
def one_match(lob1,lob2):
    return reduce(lambda x,y: y[0] and y[1] or x, zip(lob1,lob2), False)
    
#find label in dictionary fdict with epa vector closest to epa-vector oval
def findNearestEPAVector(oval,fdict,inst_list=None):
    start=True
    mind=0
    for f in fdict:
        if inst_list==None or one_match(fdict[f]["institutions"][2:11],inst_list):
            fd = fdist(oval,fdict[f])
            if start or fd<mind:
                mind=fd
                bestf=f
                start=False
    return bestf

#-----------------------------------------------------------------------------
# class to do the impression formation equations
#-----------------------------------------------------------------------------
class TauMappings:
    def __init__ (self, tdyn, tvars):
        self.gTemplateTbe = [];
        self.gTemplateTbp = [];
        self.gTemplateTba = [];
        self.gTemplateTau = [];

        self.mTbe = [];
        self.mTbp = [];
        self.mTba = [];
        self.mTau = [];

        self.tau = NP.zeros(9);

        self.H = NP.ndarray(shape=(3,9));
        self.C = NP.ndarray(shape=(1,9));

        for col in tdyn:
            if has_variable(col[0], "Tbe"):
               self.gTemplateTbe += [map(lambda x: tvars.index(x), filter(lambda y: y != "Tbe" , col[0]))]
               self.mTbe += [col[1]];
            elif has_variable(col[0], "Tbp"):
               self.gTemplateTbp += [map(lambda x: tvars.index(x), filter(lambda y: y != "Tbp" , col[0]))]
               self.mTbp += [col[1]];
            elif has_variable(col[0], "Tba"):
               self.gTemplateTba += [map(lambda x: tvars.index(x), filter(lambda y: y != "Tba" , col[0]))]
               self.mTba += [col[1]];
            else:
               self.gTemplateTau += [map(lambda x: tvars.index(x) , col[0])]
               self.mTau += [col[1]];

    def computeG(self, tau):
            self.gTbe = map(lambda i: getFactor(i, tau), self.gTemplateTbe)
            self.gTbp = map(lambda i: getFactor(i, tau), self.gTemplateTbp)
            self.gTba = map(lambda i: getFactor(i, tau), self.gTemplateTba)
            self.gTau = map(lambda i: getFactor(i, tau), self.gTemplateTau)

    def getHC(self, tau):
            self.tau = tau;

            self.gTbe = map(lambda i: getFactor(i, tau), self.gTemplateTbe)
            self.gTbp = map(lambda i: getFactor(i, tau), self.gTemplateTbp)
            self.gTba = map(lambda i: getFactor(i, tau), self.gTemplateTba)
            self.gTau = map(lambda i: getFactor(i, tau), self.gTemplateTau)

            self.H[0] = NP.dot(self.gTbe, self.mTbe)
            self.H[1] = NP.dot(self.gTbp, self.mTbp)
            self.H[2] = NP.dot(self.gTba, self.mTba)
            self.C[0] = NP.dot(self.gTau, self.mTau)

            return (self.H, self.C)
        
#------------------------------------------------------------------------------------
#reads in sentiments for gender from fbfname and returns a dictionary
#------------------------------------------------------------------------------------
def readSentiments(fbfname,gender):
    #open the input file for fundamentals of behaviours or identities
    addto_agent=0
    if gender=="female":
        addto_agent=3
    fsentiments_agent={}
    fb=file(fbfname,"rU")
    for x in set(fb):
        y=[re.sub(r"\s+","_",i.strip()) for i in x.split(",")]
        #sentiment_bins include the end-points -4.3 and 4.3 so we strip them off.
        z=["fb"+str(j) for j in NP.digitize(y[1:-1],sentiment_bins[1:-1])]
        #1,2,3 areEPA for males, 4,5,6 are EPA for females
        #for males
        fsentiments_agent[y[0]]={"elabel":z[0+addto_agent],"e":y[1+addto_agent],
                                 "plabel":z[1+addto_agent],"p":y[2+addto_agent],
                                 "alabel":z[2+addto_agent],"a":y[3+addto_agent]}
        if len(y) > 7:
            institution_list  = map(lambda i: bool(int(i)),list(re.sub(r"_","",y[7])))
            fsentiments_agent[y[0]]["institutions"]=institution_list

    fb.close()
    return fsentiments_agent


def main(argv):
    helpstring="Simple command-line ACT event simulator.\n Uage: python actsimulator.py -e <actor emotion [none]> -a <actor identity [student]> -b <actor behaviour [compromise with]> -m <object emotion [none]> -c <object identity [student]> -g <gender [male]>\n\t All flagged input arguments are labels from the dictionaries for idenitites ("+fifname+"), behaviours ("+fbfname+"), and modifiers (emotions, "+fmfname+").\n\t Change these files in the code at the top if you need to."

    #this is an example of an A-B-O combination -- default as in the paper, section 5.1
    gender = "male"
    actor="student"
    behaviour="compromise with"
    client = "student"          #this is 'object' but that is a python keyword, so we use 'client'
    emotion_actor=""
    emotion_client=""

    try:
        opts, args = getopt.getopt(argv[1:],"he:a:b:m:c:g:",["help","e=","a=","b=","m=","c=","g="])
    except getopt.GetoptError:
        print helpstring
        sys.exit(2)

    if opts==[]:
        print helpstring
        sys.exit(2)
    gender = "male"
    for opt, arg in opts:
        if opt == '-h':
            print helpstring
            sys.exit()
        elif opt == '-e':
            emotion_actor = arg
        elif opt == "-a":
            actor = arg
        elif opt == "-b":
            behaviour = arg
        elif opt == "-m":
            emotion_client = arg
        elif opt == "-c":
            client = arg
        elif opt == "-g":
            gender = arg
            


    #At this point you should have actor, behaviour, client and gender as strings, plus possibly modifiers (emotions)
    print emotion_actor,actor, behaviour,emotion_client,client, gender
        
    #must remove whitespaces 
    behaviour =  re.sub(r"\s+","_",behaviour.strip())
    actor = re.sub(r"\s+","_",actor.strip())
    client = re.sub(r"\s+","_",client.strip())
    emot_actor = re.sub(r"\s+","_",emotion_actor.strip())
    emot_client = re.sub(r"\s+","_",emotion_client.strip())
    
    epaindex = ['e','p','a']

    ## read in dictionaries 
    behaviours_agent = readSentiments(fbfname,gender)
    emotions_agent = readSentiments(fmfname,gender)
    identities_agent = readSentiments(fifname,gender)
    
    ## find epa values for relevant entities
    behaviour_epa = map(lambda x: float(behaviours_agent[behaviour][x]),epaindex)
    actor_epa = map(lambda x: float(identities_agent[actor][x]),epaindex)
    client_epa = map(lambda x: float(identities_agent[client][x]),epaindex)
    if not emotion_actor=="":
        emotion_actor_epa = map(lambda x: float(emotions_agent[emot_actor][x]),epaindex)
    if not emotion_client=="":
        emotion_client_epa = map(lambda x: float(emotions_agent[emot_client][x]),epaindex)

    ## compute the mappings needed for impression formation
    agentMappings=TauMappings(tdyn_male, tvars)
    
    ## compute modified identity for actor
    if not emotion_actor=="":
        modid_actor=computeModifiedIdentity(actor_epa,emotion_actor_epa);
    else:
        modid_actor=actor_epa
    print "(possibly) modified identity for actor: ",modid_actor
    
    ## compute modified identity for object
    if not emotion_client=="":
        modid_client=computeModifiedIdentity(client_epa,emotion_client_epa);
    else:
        modid_client=client_epa
    print "(possibly) modified identity for object: ",modid_client
        
    ## compute emotion for actor (possibly modified)
    if not emotion_actor=="":
        emotion=emotion_actor_epa;
    else:
        emotion=computeEmotion(modid_actor,modid_actor);
        
    print "agent emotion: ",emotion
    print findNearestEPAVector(emotion,emotions_agent);
    
    ## compute emotion for object if not specified
    if not emotion_client=="":
        emotion=emotion_client_epa;
    else:
        emotion=computeEmotion(modid_client,modid_client);
    print "object emotion",emotion
    print findNearestEPAVector(emotion,emotions_agent);
    
    ## create fundamental sentiment vector
    f = NP.hstack([modid_actor,behaviour_epa,modid_client])
    print "fundamental sentiment: ",f
    
    #tau starts equal to f
    tau = f
    
    ## compute transient impression
    (h, c) = agentMappings.getHC(tau);
    
    impression_tmp = NP.dot(h.transpose(),f[3:6])+c
    impression = impression_tmp[0]
    
    print "transient impression: ",impression
    
    ## compute total deflection
    deflection = NP.dot(impression-f,impression-f)
    print "deflection: ",deflection

    ## compute actor deflection
    #deflection = NP.dot(impression[0:3]-f[0:3],impression[0:3]-f[0:3])
    #print "actor deflection: ",deflection

    ## compute transient emotion for actor
    emotion=computeEmotion(f[0:3],impression[0:3]);
    print "post event agent emotion: ",emotion
    print findNearestEPAVector(emotion,emotions_agent);
    
    ## compute transient emotion for object
    emotion=computeEmotion(f[6:9],impression[6:9]);
    print "post event client emotion: ",emotion
    print findNearestEPAVector(emotion,emotions_agent);

if __name__ == "__main__":
    main(sys.argv)
        
        
