%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%  Code for replication of results in the paper:
%%%%  Denotative and Connotative Control of Uncertainty: A Computational Dual Process Model
%%%%  Authors: Jesse Hoey (University of Waterloo),  
%%%%           Neil MacKinnon (University of Guelph),
%%%%           Tobias Schroeder  (Potdsam University of Applied Sciences)
%%%%  Copyright: the authors, 2021
%%%%  For research purposes only.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% This file:  main script to generate all Figures 2-4
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% the EPA value and covariance matrices for the identities 'nurse' and 'doctor' from the USA 2015 survey
%% all datasets available by following links from bayesact.ca or at
%% http://affectcontroltheory.org/resources-for-researchers/data-sets-for-simulation/
%% note that these EPA ratings are used for demonstration purposes only and could be replaced with any two identities
%% first three elements are the mean E,P,A,
%% last nine elements are the covariance matrix 
nurseepa=[2.8589,1.8939,0.6471,1.9354,0.9856,0.7095,0.9856,2.4594,0.9377,0.7095,0.9377,3.6225];
doctorepa=[2.7207,2.9530,0.2312,1.6237,0.8382,0.6504,0.8382,1.4233,0.3568,0.6504,0.3568,5.0821];

%% power dimension is the one under consideration
%% if the evaluation dimension is needed instead, change to 1
%% if activity, change to 3
dim_to_use = 2
dim_labels={'evaluation';'power';'activity'};
%% flag to print all to PDFs if needed
%% if output_file_ext is 'none' we don't print out
output_file_ext='eps'  % can be 'none','eps','jpg','png','pdf'...
print_flag=output_file_ext;
print_to_file=not(strcmp(print_flag,'none'))
if output_file_ext=='eps'
  print_flag='epsc';  %% for color
end


%% flag to use actual covariance values for figure 2 and 4
%% SET TO FALSE to get the exact figures shown in the paper
use_real_cov=false;

%% manually set normalization factor for the prior over y - used only for figure 2
use_manual_pynorm_fig2=true
%% this was done only for visualization purposes for the paper as the figure is cut off at y=5
priory_norm=20.0;

%% mean of the prior on y, but will loop over this below
uyorig=3.0;
uy=uyorig;

%% variance of the prior on y (is the square of this) - this is standard deviation
sigyorig=1.0;
sigy=sigyorig;

%% variance in the somatic transform (is the square of this) - this is std. deviation
%% this is chosen as fixed, although it may be dependent on x, and could therefore be measured
%% For example, the variance in the power measurement for 'nurse' is about 2.5
%% while for doctor it is 1.5.
%% in order to 
%% Figure 3 varies this to examine how the results are dependent on it, but again keeps it
%% constant across x to focus on the primary objective in the paper
gamma=[0.3,0.3];
%% use this version to generate figures using the actual variances from the USA 2015 dataset:
if use_real_cov
  gamma=[sqrt(nurseepa(4*dim_to_use)),sqrt(doctorepa(4*dim_to_use))];
end

%% probability that the person is nurse
pxorig=0.7;
pxporig=[pxorig,1-pxorig];

%% prior denotative distribution
pxp = pxporig;

%% power values of nurse and doctor (mean ratings, USA 2015 dataset)
nursep=nurseepa(dim_to_use);
doctorp=doctorepa(dim_to_use);

%% combined
xdist=[nursep,doctorp];



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%%%% Figure 2
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
priorys=[];
pprimeys=[];
pprimexs=[];
%% prior means in y
uys=-1:1.0:5;
%% spread over y
y=[-4.95:0.05:5];
%% loop over those means
for uy=uys
  %% compute posteriors
  if use_manual_pynorm_fig2
    [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y,priory_norm);
  else
    [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y);
  end

  %% collect information 
  priorys=[priorys;priory];
  pprimeys=[pprimeys;pprimey];
  pprimexs=[pprimexs;pprimex];

end
uy=uyorig;

nuys=size(priorys,1);

%% now plot as a function of y for different values of mean of the prior over y, uy
%% this is figure 2a in the paper
figure(1);
clf;
hold off;
plot(repmat(y,nuys,1)',pprimeys','LineWidth',2);
legendstring=repmat([" u_y = "],nuys,1)+string(uys)'+repmat(["    S(P')="],nuys,1)+num2str(sum(-pprimexs.*log2(pprimexs),2),'%1.2f')+repmat(["    P'(nurse)="],nuys,1)+num2str(pprimexs(:,1),2);
legend(legendstring,'AutoUpdate','off','Location','northwest');
hold on;
set(gca,'ColorOrderIndex',1)
plot(repmat(y,nuys,1)',priorys','--','LineWidth',2);
set(gca,'FontWeight', 'bold');
ylabel('P(y)','FontSize',12,'FontWeight','bold');
xlabel(['Y=',dim_labels{dim_to_use}],'FontSize',12,'FontWeight','bold');
title('Figure 2a');

if print_to_file
  print(['figure2a.',output_file_ext],['-d',print_flag]);
end

%% and also the results shown as a single curve of entropy and the posterior in x
%% Figure 2b in the paper
figure(2);
clf;
hold off
%reverse everything for plotting
pprimexsr=1-pprimexs;
yyaxis left;
plot(uys',sum(-pprimexsr.*log2(pprimexsr),2),'b-o','LineWidth',3);
hold on;
yyaxis right;
plot(uys',pprimexsr(:,1),'r-*','LineWidth',3);
legend(["entropy of posterior";"P'(doctor) (posterior belief its a doctor)"],'Location','northeast');
set(gca,'FontWeight', 'bold');
yyaxis left;
ylabel('Entropy','FontSize',16,'FontWeight','bold');
yyaxis right;
ylabel('Probability (doctor)','FontSize',16,'FontWeight','bold');
xlabel(['\mu_y (prior belief in ',dim_labels{dim_to_use},' of this person)'],'FontSize',12,'FontWeight','bold');
title('Figure 2b');

if print_to_file
  print(['figure2b.',output_file_ext],['-d',print_flag]);
end




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%%%% Figure 3
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
uy=uyorig;
%% define these as fixed now,
uy=3.0;
sigy=1.0;
priorys=[];
pprimeys=[];
pprimexs=[];
%% spread of y
y=[-4.95:0.05:5];
%% loop over gamma
gammas=0.2:0.5:2.5;
for gamma=gammas
  %% same as above for figure 2
  [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y);
  %% collect data
  priorys=[priorys;priory];
  pprimeys=[pprimeys;pprimey];
  pprimexs=[pprimexs;pprimex];
end
nuys=size(priorys,1);


%% now plot as a function of y
%% Figure 3a in paper
figure(3);
clf;
hold off;
plot(repmat(y,nuys,1)',pprimeys','LineWidth',2);
legendstring=repmat([" \gamma = "],nuys,1)+string(gammas)'+repmat(["      S(P')="],nuys,1)+num2str(sum(-pprimexs.*log2(pprimexs),2),'%1.2f')+repmat(["       P'(nurse)="],nuys,1)+num2str(pprimexs(:,1),2);
legend(legendstring,'AutoUpdate','off','Location','northwest');
hold on;
set(gca,'ColorOrderIndex',1);
plot(repmat(y,nuys,1)',priorys','--','LineWidth',2);
set(gca,'FontWeight', 'bold');
ylabel('P(Y)','FontSize',12,'FontWeight','bold');
xlabel(['Y=',dim_labels{dim_to_use}],'FontSize',12,'FontWeight','bold');
title('Figure 3a');
%%Print to file if needed
if print_to_file
  print(['figure3a.',output_file_ext],['-d',print_flag]);
end

%% plot overall trends
%% Figure 3b in paper
figure(4);
clf;
hold off;
pprimexsr=1-pprimexs;
xent=sum(-pprimexs.*log2(pprimexs),2);
yyaxis left;
plot(gammas,xent,'b-o','LineWidth',3);
hold on;
yyaxis right;
plot(gammas',pprimexsr(:,1),'r-*','LineWidth',3);
legend(["entropy of posterior";"P'(doctor) (posterior belief its a doctor)"],'Location','northeast');
set(gca,'FontWeight', 'bold');
yyaxis right;
ylabel('Probability (doctor)','FontSize',16,'FontWeight','bold','Color','r');
yyaxis left;
ylabel('Entropy','FontSize',16,'FontWeight','bold','Color','b');
xlabel('\gamma','FontSize',12,'FontWeight','bold');
title('Figure 3b');

if print_to_file
  print(['figure3b.',output_file_ext],['-d',print_flag]);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%%%% Figure 4
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%% diferent priors on x
%% fixed values
uy=3.5;
sigy=1.0;
%% for visualization purposes, we select a smaller gamma -- this is what is done in the paper
gamma=[0.2,0.2];
%% could also use real values here - 
if use_real_cov
  gamma=[sqrt(nurseepa(4*dim_to_use)),sqrt(doctorepa(4*dim_to_use))];
end
priorys=[];
pprimeys=[];
pprimexs=[];
%% loop over different values for the prior over x
pxs=[0.5:0.1:0.9,0.95,0.99];
for px=pxs
  %% same as above for figure 2, now pxp changes each time
  pxp=[px,1-px];
  [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y);
  %% collect data
  priorys=[priorys;priory];
  pprimeys=[pprimeys;pprimey];
  pprimexs=[pprimexs;pprimex];
end


nuys=size(priorys,1);

%% now plot as a function of y for different values of the prior over x, ux
%% Figure 4a
figure(5);
clf;
hold off;
plot(repmat(y,nuys,1)',pprimeys','LineWidth',2);
hold on;
xent=-pxs.*log2(pxs)-(1-pxs).*log2(1-pxs);
legendvals=num2cell(string(pxs));
legendstring=repmat(["P(nurse)= "],size(pxs,2),1)+legendvals'+repmat(["     S(P)= "],size(pxs,2),1)+num2str(xent','%1.2f')+repmat(["    S(P')="],nuys,1)+num2str(sum(-pprimexs.*log2(pprimexs),2),'%1.2f')+repmat(["     P'(nurse)="],nuys,1)+num2str(pprimexs(:,1),2);
legend(legendstring,'AutoUpdate','off','Location','northwest');
set(gca,'ColorOrderIndex',1);
%% priorys are all the same though - maybe plot in black
plot(repmat(y,nuys,1)',priorys','--','LineWidth',2,'color','black');
set(gca,'FontWeight', 'bold');
ylabel('P(Y)','FontSize',12,'FontWeight','bold');
xlabel(['Y=',dim_labels{dim_to_use}],'FontSize',12,'FontWeight','bold');
title('Figure 4a');
if print_to_file
  print(['figure4a.',output_file_ext],['-d',print_flag]);
end


%%plot trends for prior and posterior entropy as a function of the prior over x ,ux
%% Figure 4b
figure(6);
clf;
hold off;
pprimexsr=1-pprimexs;
xentp=sum(-pprimexs.*log2(pprimexs),2);
yyaxis left;
plot(pxs,xent,'k--s','LineWidth',3);
hold on;
plot(pxs,xentp,'b-o','LineWidth',3);
yyaxis right;
plot(pxs,pprimexsr(:,1),'r-*','LineWidth',3);
legend(["entropy of prior in x";"entropy of posterior in x";"P'(doctor) (posterior belief its a doctor)"],'Location','northeast');
set(gca,'FontWeight', 'bold');
yyaxis right;
ylabel('Probability (doctor)','FontSize',16,'FontWeight','bold','Color','r');
yyaxis left;
ylabel('Entropy','FontSize',16,'FontWeight','bold','Color','b');
xlabel('P(X=nurse)','FontSize',12,'FontWeight','bold');
title('Figure 4b');
if print_to_file
  print(['figure4b.',output_file_ext],['-d',print_flag]);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%%%% Additional BONUS Figure not in the paper 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%55
%% now change sigma_y for a fixed u_y=4.0;
priorys=[];
pprimeys=[];
pprimexs=[];
uy=3.0;
gamma=[0.2,0.2];
if use_real_cov
  gamma=[sqrt(nurseepa(4*dim_to_use)),sqrt(doctorepa(4*dim_to_use))];
end
%% reset prior in x
pxp=pxporig;
%% loop over sigma_y values
sigys=[0.1,0.5,1.0,2.0];
%% again, same loop as above
for sigy=sigys
    %% same as above for figure 2,
  if use_manual_pynorm_fig2
    [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y,priory_norm);
  else
    [priory,pprimey,pprimex]=computePost(uy,sigy,pxp,xdist,gamma,y);
  end
  %% collect data
  priorys=[priorys;priory];
  pprimeys=[pprimeys;pprimey];
  pprimexs=[pprimexs;pprimex];
end

nuys=size(priorys,1);
nsigys=size(sigys);

%% now plot as a function of y for different values of the prior variance over y, \sigma_y
figure(7);
clf;
hold off;
plot(repmat(y,nuys,1)',pprimeys','LineWidth',2);
legendstring=repmat([" \sigma_y = "],[nsigys(:,2),1])+string(sigys)'+repmat(["    S(P')="],[nuys,1])+num2str(sum(-pprimexs.*log2(pprimexs),2),'%1.2f')+repmat(["    P'(nurse)="],[nuys,1])+num2str(pprimexs(:,1),2);
legend(legendstring,'AutoUpdate','off','Location','northwest');
hold on;
set(gca,'ColorOrderIndex',1);
plot(repmat(y,nuys,1)',priorys','--','LineWidth',2);
set(gca,'FontWeight', 'bold');
ylabel('P(y)','FontSize',12,'FontWeight','bold');
xlabel(['Y=',dim_labels{dim_to_use}],'FontSize',12,'FontWeight','bold');
title('extra figure a');
if print_to_file
  print(['figure_extra_a.',output_file_ext],['-d',print_flag]);
end

%% and also the results in a single plot for presentation
figure(8);
clf
hold off
%reverse everything
pprimexsr=1-pprimexs;
yyaxis right;
plot(sigys',sum(-pprimexsr.*log2(pprimexsr),2),'b-o','LineWidth',3);
hold on
yyaxis right;
plot(sigys',pprimexsr(:,1),'r-*','LineWidth',3);
legend(["entropy of posterior";"P'(doctor) (posterior belief its a doctor)"],'Location','northeast');
set(gca,'FontWeight', 'bold');
yyaxis right;
ylabel('Probability (doctor)','FontSize',16,'FontWeight','bold','Color','r');
yyaxis left;
ylabel('Entropy','FontSize',16,'FontWeight','bold','Color','b');
xlabel(['\sigma_y (prior uncertainty about ',dim_labels{dim_to_use},' of this person)'],'FontSize',12,'FontWeight','bold');

title('extra figure b');
if print_to_file
  print(['figure_extra_b.',output_file_ext],['-d',print_flag]);
end

